-- FBatch5.sql
-- Purpose: Waits for etpfetrt to be populated by ETPFet.exe /SQL, updates etpfetbt, logs results
-- Version: v250725d

DO $$
DECLARE
    rt_count BIGINT;
    retry_count INTEGER := 0;
    max_retries INTEGER := 60; -- 5 minutes at 5-second intervals
    retry_delay INTEGER := 5; -- seconds
    log_message TEXT;
BEGIN
    -- Verify etpfetrt accessibility
    IF NOT EXISTS (
        SELECT FROM information_schema.tables 
        WHERE table_schema = 'public' AND table_name = 'etpfetrt'
    ) THEN
        log_message := 'FBatch5: etpfetrt table does not exist.';
        INSERT INTO public.etp_log (log_time, log_message)
        VALUES (TO_CHAR(CURRENT_TIMESTAMP, 'YYYY-MM-DD HH24:MI:SS'), log_message);
        RAISE EXCEPTION '%', log_message;
    END IF;

    -- Log table found
    INSERT INTO public.etp_log (log_time, log_message)
    VALUES (TO_CHAR(CURRENT_TIMESTAMP, 'YYYY-MM-DD HH24:MI:SS'), 'FBatch5: etpfetrt table found, starting poll');

    -- Loop to check etpfetrt
    LOOP
        retry_count := retry_count + 1;
        BEGIN
            SELECT COUNT(*) INTO rt_count FROM public.etpfetrt;
        EXCEPTION
            WHEN OTHERS THEN
                log_message := format('FBatch5: Error querying etpfetrt: %s', SQLERRM);
                INSERT INTO public.etp_log (log_time, log_message)
                VALUES (TO_CHAR(CURRENT_TIMESTAMP, 'YYYY-MM-DD HH24:MI:SS'), log_message);
                RAISE EXCEPTION '%', log_message;
        END;
        log_message := format('FBatch5: Retry %s/%s: etpfetrt has %s rows', retry_count, max_retries, rt_count);
        INSERT INTO public.etp_log (log_time, log_message)
        VALUES (TO_CHAR(CURRENT_TIMESTAMP, 'YYYY-MM-DD HH24:MI:SS'), log_message);
        IF rt_count > 0 THEN
            log_message := format('FBatch5: etpfetrt contains %s rows. Processing complete.', rt_count);
            INSERT INTO public.etp_log (log_time, log_message)
            VALUES (TO_CHAR(CURRENT_TIMESTAMP, 'YYYY-MM-DD HH24:MI:SS'), log_message);
            -- Embedded FBatch7.sql logic
            DECLARE
                current_batch_oid BIGINT;
                end_time TIMESTAMP := CURRENT_TIMESTAMP;
                start_time VARCHAR(30);
                execution_time TEXT;
            BEGIN
                SELECT MAX(oid) INTO current_batch_oid FROM public.etpfetbt WHERE Status = 'Pending';
                IF current_batch_oid IS NULL THEN
                    log_message := 'FBatch5: No Pending batch found in etpfetbt.';
                    INSERT INTO public.etp_log (log_time, log_message)
                    VALUES (TO_CHAR(CURRENT_TIMESTAMP, 'YYYY-MM-DD HH24:MI:SS'), log_message);
                ELSE
                    SELECT StartTime INTO start_time FROM public.etpfetbt WHERE oid = current_batch_oid;
                    IF start_time IS NULL THEN
                        log_message := format('FBatch5: Start time not found for batch OID: %s', current_batch_oid);
                        INSERT INTO public.etp_log (log_time, log_message)
                        VALUES (TO_CHAR(CURRENT_TIMESTAMP, 'YYYY-MM-DD HH24:MI:SS'), log_message);
                    ELSE
                        execution_time := TO_CHAR(end_time - start_time::TIMESTAMP, 'HH24:MI:SS');
                        UPDATE public.etpfetbt
                        SET EndTime = TO_CHAR(end_time, 'YYYY-MM-DD HH24:MI:SS'),
                            ExecutionTime = execution_time,
                            Status = 'Completed'
                        WHERE oid = current_batch_oid;
                        log_message := format('FBatch5: Batch OID %s marked as Completed. EndTime: %s, ExecutionTime: %s',
                                             current_batch_oid, TO_CHAR(end_time, 'YYYY-MM-DD HH24:MI:SS'), execution_time);
                        INSERT INTO public.etp_log (log_time, log_message)
                        VALUES (TO_CHAR(CURRENT_TIMESTAMP, 'YYYY-MM-DD HH24:MI:SS'), log_message);
                    END IF;
                END IF;
            EXCEPTION
                WHEN OTHERS THEN
                    log_message := format('FBatch5: Error updating etpfetbt: %s', SQLERRM);
                    INSERT INTO public.etp_log (log_time, log_message)
                    VALUES (TO_CHAR(CURRENT_TIMESTAMP, 'YYYY-MM-DD HH24:MI:SS'), log_message);
            END;
            EXIT;
        END IF;
        IF retry_count >= max_retries THEN
            log_message := format('FBatch5: Max retries (%s) reached. etpfetrt still empty.', max_retries);
            INSERT INTO public.etp_log (log_time, log_message)
            VALUES (TO_CHAR(CURRENT_TIMESTAMP, 'YYYY-MM-DD HH24:MI:SS'), log_message);
            RAISE EXCEPTION '%', log_message;
        END IF;
        PERFORM pg_sleep(retry_delay); -- Delay in seconds
    END LOOP;
END $$;